/*
    common - Generic support for video devices

    No Copyright by Pauline Middelink <middelin@polyware.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <linux/mm.h>
#include <linux/pci.h>
#include <linux/wrapper.h>
#include <asm/io.h>
#ifdef CONFIG_BIGPHYS_AREA
#include <linux/bigphysarea.h>
#endif

#include "zr36120.h"
#include "common.h"

/*******************************/
/* Memory management functions */
/*******************************/

/* convert virtual user memory address to physical address */
/* (virt_to_phys only works for kmalloced kernel memory) */

inline unsigned long uvirt_to_phys(unsigned long adr)
{
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *ptep, pte;

	pgd = pgd_offset(current->mm, adr);
	if (pgd_none(*pgd))
		return 0;
	pmd = pmd_offset(pgd, adr);
	if (pmd_none(*pmd))
		return 0;
	ptep = pte_offset(pmd, adr/*&(~PGDIR_MASK)*/);
	pte = *ptep;
	if(pte_present(pte))
		return
		  virt_to_phys((void *)(pte_page(pte)|(adr&(PAGE_SIZE-1))));
	return 0;
}

void* rvmalloc(unsigned long size)
{
	void * mem;
	unsigned long adr, page;

	mem=vmalloc(size);
	if (mem)
	{
		adr=(unsigned long) mem;
		while (size > 0)
		{
			page = kvirt_to_phys(adr);
			mem_map_reserve(MAP_NR(phys_to_virt(page)));
			adr+=PAGE_SIZE;
			size-=PAGE_SIZE;
		}
	}
	return mem;
}

void rvfree(void* mem, unsigned long size)
{
	unsigned long adr, page;

	if (mem)
	{
		adr=(unsigned long) mem;
		while (size > 0)
		{
			page = kvirt_to_phys(adr);
			mem_map_unreserve(MAP_NR(phys_to_virt(page)));
			adr+=PAGE_SIZE;
			size-=PAGE_SIZE;
		}
		vfree(mem);
	}
}

/*******************************/
/* Memory management functions */
/*******************************/
void* bmalloc(unsigned long size)
{
	void* mem;
#ifdef CONFIG_BIGPHYS_AREA
	mem = bigphysarea_alloc_pages(size/PAGE_SIZE, 1, GFP_KERNEL);
#else
	mem = kmalloc(size, GFP_KERNEL);
#endif
	if (mem) {
		ulong adr = (ulong)mem;
		while (size > 0) {
			mem_map_reserve(MAP_NR(phys_to_virt(adr)));
			adr += PAGE_SIZE;
			size -= PAGE_SIZE;
		}
	}
	return mem;
}

void bfree(void* mem, unsigned long size)
{
	if (mem) {
		ulong adr = (ulong)mem;
		while (size > 0) {
			mem_map_unreserve(MAP_NR(phys_to_virt(adr)));
			adr += PAGE_SIZE;
			size -= PAGE_SIZE;
		}
#ifdef CONFIG_BIGPHYS_AREA
		bigphysarea_free_pages(mem);
#else
		kfree(mem);
#endif
	}
}

/****************************************************************/
struct vidbases
{
        unsigned short vendor, device;
        char *name;
        uint badr;
};

static struct vidbases vbs[] = {
        { PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_210888GX,
                "ATI MACH64 Winturbo", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_ATI, PCI_DEVICE_ID_ATI_215GT,
                "ATI MACH64 GT", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_CIRRUS, 0, "Cirrus Logic", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_DEC, PCI_DEVICE_ID_DEC_TGA,
                "DEC DC21030", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_MATROX, PCI_DEVICE_ID_MATROX_MIL,
                "Matrox Millennium", PCI_BASE_ADDRESS_1},
        { PCI_VENDOR_ID_MATROX, PCI_DEVICE_ID_MATROX_MIL_2,
                "Matrox Millennium II", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_MATROX, PCI_DEVICE_ID_MATROX_MIL_2_AGP,
                "Matrox Millennium II AGP", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_MATROX, 0x051a, "Matrox Mystique", PCI_BASE_ADDRESS_1},
        { PCI_VENDOR_ID_N9, PCI_DEVICE_ID_N9_I128,
                "Number Nine Imagine 128", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_S3, 0, "S3", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_SI, 0, "SiS", PCI_BASE_ADDRESS_0},
        { PCI_VENDOR_ID_TSENG, 0, "TSENG", PCI_BASE_ADDRESS_0},
};


/* DEC TGA offsets stolen from XFree-3.2 */

static uint dec_offsets[4] = {
        0x200000,
        0x804000,
        0,
        0x1004000
};

#define NR_CARDS (sizeof(vbs)/sizeof(struct vidbases))

/* Scan for PCI display adapter
   if more than one card is present the last one is used for now */

int find_vga(unsigned int* vidaddr)
{
        unsigned short badr;
        int found = 0, i, tga_type;
        unsigned int vidadr=0;
        struct pci_dev *dev;

        for (dev = pci_devices; dev != NULL; dev = dev->next)
        {
                if (dev->class != PCI_CLASS_NOT_DEFINED_VGA &&
                        (dev->class) >> 16 != PCI_BASE_CLASS_DISPLAY)
                {
                        continue;
                }
                if (PCI_FUNC(dev->devfn) != 0)
                        continue;

                badr=0;
                for (i=0; i<NR_CARDS; i++)
                {
                        if (dev->vendor == vbs[i].vendor)
                        {
                                if (vbs[i].device)
                                        if (vbs[i].device!=dev->device)
                                                continue;
                                DEBUG(printk(KERN_DEBUG "%s.\n", vbs[i].name));
                                badr=vbs[i].badr;
                                break;
                        }
		}
		if (!badr)
		{
			printk(KERN_ERR "zoran: Unknown video memory base address.\n");
			continue;
		}
		pci_read_config_dword(dev, badr, &vidadr);
		if (vidadr & PCI_BASE_ADDRESS_SPACE_IO)
		{
			printk(KERN_ERR "zoran: Memory seems to be I/O memory.\n");
			printk(KERN_ERR "zoran: Check entry for your card type in zoran.c vidbases struct.\n");
			continue;
		}
		vidadr &= PCI_BASE_ADDRESS_MEM_MASK;
		if (!vidadr)
		{
			printk(KERN_ERR "zoran: Memory @ 0, must be something wrong!\n");
			continue;
		}
		if (dev->vendor == PCI_VENDOR_ID_DEC &&
		dev->device == PCI_DEVICE_ID_DEC_TGA)
		{
			tga_type = (readl((unsigned long)vidadr) >> 12)
& 0x0f;
			if (tga_type != 0 && tga_type != 1 && tga_type != 3)
			{
				printk(KERN_ERR "zoran: TGA type (0x%x) unrecognized!\n", tga_type);
				found--;
			}
			vidadr+=dec_offsets[tga_type];
		}
		DEBUG(printk(KERN_DEBUG "zoran: memory @ 0x%08x, devfn: 0x%04x.\n", vidadr,dev->devfn));

		found++;
        }

        if (vidmem)
        {
                vidadr=vidmem<<20;
                printk(KERN_INFO "zoran: Video memory override: 0x%08x\n", vidadr);
                found=1;
        }

	*vidaddr = vidadr;
        return found;
}

#define  TRITON_PCON		   0x50 
#define  TRITON_BUS_CONCURRENCY   (1<<0)
#define  TRITON_STREAMING	  (1<<1)
#define  TRITON_WRITE_BURST	  (1<<2)
#define  TRITON_PEER_CONCURRENCY  (1<<3)
  
void handle_chipset(void)
{
	struct pci_dev *dev = NULL;
  
	/* Just in case some nut set this to something dangerous */
	if (triton1)
		triton1 = ZORAN_VDC_TRICOM;
	
	while ((dev = pci_find_device(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82441, dev))) 
	{
		unsigned char b;
		pci_read_config_byte(dev, 0x53, &b);
		DEBUG(printk(KERN_INFO "zoran: Host bridge: 82441FX Natoma, "));
		DEBUG(printk(KERN_DEBUG "bufcon=0x%02x\n",b));
	}

	while ((dev = pci_find_device(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82437, dev))) 
	{
		printk(KERN_INFO "zoran: Host bridge 82437FX Triton PIIX\n");
		triton1 = ZORAN_VDC_TRICOM;

#if 0			
		/* The ETBF bit SHOULD make all this unnecessary */
		/* 430FX (Triton I) freezes with bus concurrency on -> switch it off */

		pci_read_config_byte(dev, TRITON_PCON, &b);
		bo=b;
		DEBUG(printk(KERN_DEBUG "zoran: 82437FX: PCON: 0x%x\n",b));
		if(!(b & TRITON_BUS_CONCURRENCY)) 
		{
			printk(KERN_WARNING "zoran: 82437FX: disabling bus concurrency\n");
			b |= TRITON_BUS_CONCURRENCY;
		}
		if(b & TRITON_PEER_CONCURRENCY) 
		{
			printk(KERN_WARNING "zoran: 82437FX: disabling peer concurrency\n");
			b &= ~TRITON_PEER_CONCURRENCY;
		}
		if(!(b & TRITON_STREAMING)) 
		{
			printk(KERN_WARNING "zoran: 82437FX: disabling streaming\n");
			b |=  TRITON_STREAMING;
		}

		if (b!=bo) 
		{
			pci_write_config_byte(dev, TRITON_PCON, b); 
			printk(KERN_DEBUG "zoran: 82437FX: PCON changed to: 0x%x\n",b);
		}
#endif
	}
}
